import json
import base64
import os
import tempfile
from channels.generic.websocket import AsyncWebsocketConsumer
from openai import OpenAI
from .rag import query_knowledge_base
from .utils import get_system_prompt
from dotenv import load_dotenv

load_dotenv()
# client = OpenAI()  # Commented out - using Gemini API instead
client = None  # WebSocket consumer not currently used

class AgentConsumer(AsyncWebsocketConsumer):
    async def connect(self):
        await self.accept()
        self.history = [
            {"role": "system", "content": get_system_prompt()}
        ]

    async def disconnect(self, close_code):
        pass

    async def receive(self, text_data):
        text_data_json = json.loads(text_data)
        
        if 'audio' in text_data_json:
            audio_data = base64.b64decode(text_data_json['audio'])
            
            # 1. Save to temp file
            with tempfile.NamedTemporaryFile(delete=False, suffix=".webm") as temp_audio:
                temp_audio.write(audio_data)
                temp_audio_path = temp_audio.name

            try:
                # 2. Transcribe (STT)
                with open(temp_audio_path, "rb") as audio_file:
                    transcription = client.audio.transcriptions.create(
                        model="whisper-1", 
                        file=audio_file
                    )
                user_text = transcription.text
                print(f"User said: {user_text}")

                # 3. RAG Lookup
                rag_response = query_knowledge_base(user_text)
                
                # 4. Generate Response (LLM)
                # We inject RAG context into the message
                context_message = f"Context from Knowledge Base: {rag_response}\n\nUser Query: {user_text}"
                
                self.history.append({"role": "user", "content": context_message})
                
                completion = client.chat.completions.create(
                    model="gpt-4o",
                    messages=self.history
                )
                
                ai_text = completion.choices[0].message.content
                self.history.append({"role": "assistant", "content": ai_text})
                print(f"AI said: {ai_text}")

                # 5. Generate Audio (TTS)
                response = client.audio.speech.create(
                    model="tts-1",
                    voice="alloy",
                    input=ai_text
                )
                
                # Save TTS to temp file to read as base64
                with tempfile.NamedTemporaryFile(delete=False, suffix=".mp3") as temp_tts:
                    response.stream_to_file(temp_tts.name)
                    temp_tts_path = temp_tts.name
                
                with open(temp_tts_path, "rb") as f:
                    tts_base64 = base64.b64encode(f.read()).decode('utf-8')

                # 6. Send back
                await self.send(text_data=json.dumps({
                    'message': ai_text,
                    'audio': tts_base64
                }))

                # Cleanup
                os.remove(temp_tts_path)

            except Exception as e:
                print(f"Error: {e}")
                await self.send(text_data=json.dumps({
                    'message': f"Error: {str(e)}"
                }))
            finally:
                if os.path.exists(temp_audio_path):
                    os.remove(temp_audio_path)

