const helpBubble = document.getElementById('helpBubble');
const modalOverlay = document.getElementById('modalOverlay');
const closeModal = document.getElementById('closeModal');
const submitButton = document.getElementById('submitButton');
const userInput = document.getElementById('userInput');
const languageSelect = document.getElementById('languageSelect');
const statusText = document.getElementById('statusText');
const videoPlayer = document.getElementById('agent-video');
const modalMicButton = document.getElementById('modalMicButton');
const welcomeText = document.getElementById('welcomeText');
const answerDisplay = document.getElementById('answerDisplay');
const questionText = document.getElementById('questionText');
const answerText = document.getElementById('answerText');

let recognition = null;

if ('webkitSpeechRecognition' in window) {
    recognition = new webkitSpeechRecognition();
    recognition.continuous = false;
    recognition.interimResults = false;

    recognition.onstart = () => {
        console.log('Speech recognition started');
        statusText.textContent = 'Listening...';
    };

    recognition.onresult = (event) => {
        const transcript = event.results[0][0].transcript;
        console.log('Recognized:', transcript);
        userInput.value = transcript;
        statusText.textContent = 'Got it!';
    };

    recognition.onerror = (event) => {
        console.error('Speech recognition error:', event.error);
        modalMicButton.style.background = '#000';
        statusText.textContent = 'Error: ' + event.error;
    };

    recognition.onend = () => {
        console.log('Speech recognition ended');
        modalMicButton.style.background = '#000';
        if (!userInput.value) {
            statusText.textContent = '';
        }
    };
} else {
    console.warn('Speech recognition not supported');
}

helpBubble.addEventListener('click', () => {
    modalOverlay.classList.add('active');
    userInput.focus();
});

closeModal.addEventListener('click', () => {
    modalOverlay.classList.remove('active');
});

modalOverlay.addEventListener('click', (e) => {
    if (e.target === modalOverlay) modalOverlay.classList.remove('active');
});

modalMicButton.addEventListener('click', () => {
    console.log('Mic button clicked!');
    if (!recognition) {
        alert('Speech recognition is not supported in your browser. Please use Chrome or Edge.');
        return;
    }

    try {
        const langMap = { 'en': 'en-US', 'hi': 'hi-IN', 'no': 'nb-NO' };
        recognition.lang = langMap[languageSelect.value] || 'en-US';
        console.log('Starting recognition with language:', recognition.lang);
        recognition.start();
        modalMicButton.style.background = '#ff0000';
        statusText.textContent = 'Listening... speak now';
    } catch (error) {
        console.error('Recognition error:', error);
        alert('Microphone error: ' + error.message);
    }
});

submitButton.addEventListener('click', generateResponse);
userInput.addEventListener('keypress', (e) => {
    if (e.key === 'Enter') generateResponse();
});

// WebRTC State
let peerConnection;
let streamId;
let sessionId;
let sessionClientAnswer;

async function initStream() {
    if (peerConnection && peerConnection.connectionState === 'connected') {
        return;
    }

    statusText.textContent = "Connecting to D-ID Stream...";

    // 1. Create Peer Connection with TURN server for NAT traversal
    peerConnection = new RTCPeerConnection({
        iceServers: [
            { urls: "stun:stun.l.google.com:19302" },
            {
                urls: "turn:openrelay.metered.ca:80",
                username: "openrelayproject",
                credential: "openrelayproject"
            }
        ]
    });

    peerConnection.onicecandidate = (event) => {
        if (event.candidate) {
            fetch('/submit_ice/', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    stream_id: streamId,
                    session_id: sessionId,
                    candidate: event.candidate,
                    sdpMid: event.candidate.sdpMid,
                    sdpMLineIndex: event.candidate.sdpMLineIndex
                })
            });
        }
    };

    peerConnection.ontrack = (event) => {
        console.log("Got track:", event.streams[0]);
        videoPlayer.srcObject = event.streams[0];

        // Try to play unmuted first (in case we have user gesture token)
        videoPlayer.muted = false;
        videoPlayer.play().then(() => {
            console.log("Video playing unmuted!");
        }).catch(e => {
            console.warn("Unmuted autoplay failed, trying muted:", e);
            videoPlayer.muted = true;
            videoPlayer.play().catch(err => console.error("Muted playback also failed:", err));
        });
    };

    // Monitor connection states
    peerConnection.oniceconnectionstatechange = () => {
        console.log("ICE State:", peerConnection.iceConnectionState);
        if (peerConnection.iceConnectionState === 'connected') {
            statusText.textContent = "Connected! Waiting for you...";
            // Once connected, try to unmute if possible (might fail without interaction)
            videoPlayer.muted = false;
        } else if (peerConnection.iceConnectionState === 'failed') {
            statusText.textContent = "Connection Failed. Refreshing...";
            location.reload();
        }
    };

    peerConnection.onconnectionstatechange = () => {
        console.log("Connection State:", peerConnection.connectionState);
    };

    // 2. Create D-ID Stream
    const createResp = await fetch('/create_stream/', { method: 'POST' });
    const createData = await createResp.json();

    streamId = createData.id;
    sessionId = createData.session_id;
    const offer = createData.offer;

    // 3. Set Remote Description (Offer)
    await peerConnection.setRemoteDescription(offer);

    // 4. Create Answer
    const answer = await peerConnection.createAnswer();
    await peerConnection.setLocalDescription(answer);

    // 5. Start Stream
    await fetch('/start_stream/', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
            stream_id: streamId,
            session_id: sessionId,
            answer: answer
        })
    });

    statusText.textContent = "Connected! Ask a question.";
}

// Initialize stream on load automatically
// DISABLED: We're using Clips API now, not streaming
// initStream();

let isInitializing = false;
async function guardedInitStream() {
    if (isInitializing) return;
    isInitializing = true;
    try {
        await initStream();
    } finally {
        isInitializing = false;
    }
}

async function generateResponse() {
    const text = userInput.value.trim();
    if (!text) {
        statusText.textContent = "Please enter a question";
        return;
    }

    // Disable button immediately to prevent multiple clicks
    submitButton.disabled = true;
    submitButton.textContent = "Thinking...";
    statusText.textContent = "AI is thinking...";

    // STOP and clear old video immediately
    videoPlayer.pause();
    videoPlayer.src = "";
    videoPlayer.load();

    // Try to unmute video (non-blocking, don't await)
    videoPlayer.muted = false;
    videoPlayer.play().catch(e => {
        console.warn("Video play failed:", e);
        videoPlayer.muted = true;
        videoPlayer.play().catch(err => console.error("Muted play also failed:", err));
    });

    try {
        // Send text to backend -> LLM -> D-ID Talk
        const response = await fetch('/chat/', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRFToken': getCookie('csrftoken')
            },
            body: JSON.stringify({
                text: text,
                stream_id: streamId,
                session_id: sessionId
            })
        });

        if (!response.ok) throw new Error('Generation failed');

        const data = await response.json();

        modalOverlay.classList.remove('active');

        // Display answer
        welcomeText.style.display = 'none';
        answerDisplay.classList.add('active');
        questionText.style.display = 'none';
        answerText.textContent = data.answer;

        // Display the lip-synced video
        if (data.video_url) {
            console.log("Video URL received:", data.video_url);
            statusText.textContent = "Loading video...";

            // Set video source
            videoPlayer.src = data.video_url;
            videoPlayer.muted = false;
            videoPlayer.controls = true;
            videoPlayer.autoplay = true;
            videoPlayer.load();

            // Force play after tiny delay
            setTimeout(() => {
                videoPlayer.play().then(() => {
                    console.log("✓ Video playing!");
                    statusText.textContent = "Speaking...";
                }).catch(err => {
                    console.warn("Autoplay blocked, trying muted:", err);
                    videoPlayer.muted = true;
                    videoPlayer.play().then(() => {
                        statusText.textContent = "Playing (click to unmute)";
                        videoPlayer.addEventListener('click', () => {
                            videoPlayer.muted = false;
                            statusText.textContent = "Speaking...";
                        }, { once: true });
                    });
                });
            }, 200);

            videoPlayer.addEventListener('ended', () => {
                statusText.textContent = "Done!";
            }, { once: true });
        } else {
            console.warn("No video URL received");
            statusText.textContent = "Answer ready (no video)";
        }

        userInput.value = "";

    } catch (err) {
        console.error(err);
        statusText.textContent = "Error: " + err.message;
    } finally {
        submitButton.disabled = false;
        submitButton.textContent = "Ask Question";
    }
}

function getCookie(name) {
    let cookieValue = null;
    if (document.cookie && document.cookie !== '') {
        const cookies = document.cookie.split(';');
        for (let i = 0; i < cookies.length; i++) {
            const cookie = cookies[i].trim();
            if (cookie.substring(0, name.length + 1) === (name + '=')) {
                cookieValue = decodeURIComponent(cookie.substring(name.length + 1));
                break;
            }
        }
    }
    return cookieValue;
}
