import os
import asyncio
import edge_tts
from moviepy.editor import ImageClip, AudioFileClip
from django.conf import settings
import requests
import time

# Ensure media directories exist
MEDIA_ROOT = os.path.join(settings.BASE_DIR, 'media')
os.makedirs(MEDIA_ROOT, exist_ok=True)

VOICE_MAP = {
    'en': 'en-US-AriaNeural',
    'hi': 'hi-IN-SwaraNeural',
    'no': 'nb-NO-PernilleNeural'
}

import requests
import json

import base64

ELEVENLABS_API_KEY = os.environ.get("ELEVENLABS_API_KEY")
DID_API_KEY = os.environ.get("DID_API_KEY")

def get_did_headers():
    auth_value = DID_API_KEY
    if DID_API_KEY and ":" in DID_API_KEY:
        auth_value = base64.b64encode(DID_API_KEY.encode('utf-8')).decode('utf-8')
    return {
        "Authorization": f"Basic {auth_value}",
        "Content-Type": "application/json"
    }

# ElevenLabs Settings
ELEVENLABS_VOICE_ID = "21m00Tcm4TlvDq8ikWAM" # Rachel

def generate_elevenlabs_audio(text):
    """Generate audio using ElevenLabs and return the file path"""
    url = f"https://api.elevenlabs.io/v1/text-to-speech/{ELEVENLABS_VOICE_ID}"
    headers = {
        "Accept": "audio/mpeg",
        "Content-Type": "application/json",
        "xi-api-key": ELEVENLABS_API_KEY
    }
    data = {
        "text": text,
        "model_id": "eleven_monolingual_v1",
        "voice_settings": {
            "stability": 0.5,
            "similarity_boost": 0.5
        }
    }
    
    response = requests.post(url, json=data, headers=headers)
    if response.status_code != 200:
        raise Exception(f"ElevenLabs Error: {response.text}")
        
    filename = f"eleven_{os.urandom(4).hex()}.mp3"
    path = os.path.join(MEDIA_ROOT, filename)
    with open(path, "wb") as f:
        f.write(response.content)
        
    return path

def upload_to_fileio(file_path):
    """
    Upload file to public hosting service with multiple fallbacks.
    Returns a publicly accessible URL that D-ID can download from.
    D-ID requires public HTTP/HTTPS URLs, NOT base64 data URLs.
    """
    
    # Method 1: freeimage.host (No API key needed, most reliable)
    try:
        url = "https://freeimage.host/api/1/upload"
        with open(file_path, 'rb') as f:
            files = {'source': f}
            response = requests.post(url, files=files, timeout=15)
        
        if response.status_code == 200:
            data = response.json()
            if data.get('status_code') == 200:
                image_url = data['image']['url']
                print(f"✓ Uploaded to freeimage.host: {image_url}")
                return image_url
            else:
                print(f"freeimage.host error: {data}")
    except Exception as e:
        print(f"freeimage.host failed: {e}")
    
    # Method 2: imgbb.com (Requires free API key - optional)
    imgbb_key = os.environ.get("IMGBB_API_KEY")
    if imgbb_key:
        try:
            with open(file_path, 'rb') as f:
                image_data = base64.b64encode(f.read()).decode('utf-8')
            
            url = "https://api.imgbb.com/1/upload"
            payload = {
                'key': imgbb_key,
                'image': image_data,
            }
            response = requests.post(url, data=payload, timeout=15)
            
            if response.status_code == 200:
                data = response.json()
                if data.get('success'):
                    image_url = data['data']['url']
                    print(f"✓ Uploaded to imgbb: {image_url}")
                    return image_url
        except Exception as e:
            print(f"imgbb upload failed: {e}")
    
    # Method 3: transfer.sh (Fallback)
    try:
        filename = os.path.basename(file_path)
        url = f"https://transfer.sh/{filename}"
        
        with open(file_path, 'rb') as f:
            response = requests.put(url, data=f, timeout=10)
        
        if response.status_code == 200:
            public_url = response.text.strip()
            print(f"✓ Uploaded to transfer.sh: {public_url}")
            return public_url
        else:
            print(f"transfer.sh error: {response.status_code}")
    except Exception as e:
        print(f"transfer.sh failed: {e}")
    
    # If all methods fail, raise exception
    raise Exception("All image upload services failed. Cannot create public URL for D-ID.")

# D-ID Streaming Functions
def create_did_stream(source_url):
    """Create a D-ID Realtime Stream"""
    url = "https://api.d-id.com/talks/streams"
    headers = get_did_headers()
    data = {
        "source_url": source_url
    }
    
    print(f"Creating D-ID stream with source URL (first 100 chars): {source_url[:100]}...")
    
    response = requests.post(url, json=data, headers=headers)
    if response.status_code not in [200, 201]:
        error_msg = f"D-ID Create Stream Error: {response.status_code} - {response.text}"
        print(error_msg)
        raise Exception(error_msg)
        
    print(f"✓ D-ID Stream created successfully! Status: {response.status_code}")
    
    with open("debug_log.txt", "a", encoding="utf-8") as log:
        log.write(f"D-ID Status: {response.status_code}\n")
        log.write(f"D-ID Response: {response.text[:500]}\n")

    try:
        return response.json()
    except Exception as e:
        with open("debug_log.txt", "a", encoding="utf-8") as log:
            log.write(f"D-ID JSON Error: {e}\n")
        raise e

def start_did_stream(stream_id, session_id, answer):
    """Start the D-ID Stream with SDP Answer"""
    url = f"https://api.d-id.com/talks/streams/{stream_id}/sdp"
    headers = get_did_headers()
    
    # D-ID expects the answer in a specific format
    data = {
        "answer": answer,
        "session_id": session_id
    }
    
    print(f"Starting D-ID stream: {stream_id}")
    
    response = requests.post(url, json=data, headers=headers)
    if response.status_code not in [200, 201]:
        print(f"D-ID Start Stream Error: {response.status_code} - {response.text}")
        raise Exception(f"D-ID Start Stream Error: {response.text}")
        
    print(f"✓ D-ID Stream started successfully!")
    return response.json()

def submit_did_ice(stream_id, session_id, candidate, sdp_mid, sdp_m_line_index):
    """Submit ICE Candidate to D-ID"""
    url = "https://api.d-id.com/talks/streams/ice"
    headers = get_did_headers()
    data = {
        "stream_id": stream_id,
        "session_id": session_id,
        "candidate": candidate,
        "sdpMid": sdp_mid,
        "sdpMLineIndex": sdp_m_line_index
    }
    
    requests.post(url, json=data, headers=headers)

def send_did_talk(stream_id, session_id, text=None, audio_url=None, audio_path=None):
    """Send text or audio to D-ID stream to make avatar talk.
    Uses D-ID's built-in TTS if text is provided (most reliable).
    """
    url = f"https://api.d-id.com/talks/streams/{stream_id}"
    headers = get_did_headers()
    
    # Use text-based TTS (most reliable, no upload needed)
    if text:
        data = {
            "script": {
                "type": "text",
                "input": text,
                "provider": {
                    "type": "microsoft",
                    "voice_id": "en-US-JennyNeural"
                }
            },
            "driver_url": "bank://lively/",
            "config": {
                "stitch": True
            },
            "session_id": session_id
        }
        print(f"Sending text to D-ID for TTS: {text[:50]}...")
    elif audio_url:
        # Use URL if provided
        data = {
            "script": {
                "type": "audio",
                "audio_url": audio_url
            },
            "driver_url": "bank://lively/",
            "config": {
                "stitch": True
            },
            "session_id": session_id
        }
        print(f"Sending audio URL to D-ID: {audio_url}")
    else:
        raise Exception("Either text or audio_url must be provided")
    
    response = requests.post(url, json=data, headers=headers)
    if response.status_code not in [200, 201]:
        print(f"D-ID Talk Error: {response.status_code} - {response.text}")
        raise Exception(f"D-ID Talk Error: {response.text}")
    
    print(f"✓ D-ID Talk sent successfully!")
    return response.json()


async def generate_audio(text, lang, output_filename):
    voice = VOICE_MAP.get(lang, 'en-US-AriaNeural')
    communicate = edge_tts.Communicate(text, voice)
    output_path = os.path.join(MEDIA_ROOT, output_filename)
    await communicate.save(output_path)
    return output_path

def generate_subtitles(text, duration, output_filename):
    words = text.split()
    word_count = len(words)
    words_per_second = word_count / duration
    
    chunk_size = max(1, int(words_per_second * 3))
    chunks = []
    for i in range(0, word_count, chunk_size):
        chunks.append(" ".join(words[i:i+chunk_size]))
    
    vtt_content = "WEBVTT\n\n"
    start_time = 0
    chunk_duration = duration / len(chunks)
    
    for i, chunk in enumerate(chunks):
        end_time = start_time + chunk_duration
        start_fmt = format_timestamp(start_time)
        end_fmt = format_timestamp(end_time)
        vtt_content += f"{start_fmt} --> {end_fmt}\n{chunk}\n\n"
        start_time = end_time
        
    output_path = os.path.join(MEDIA_ROOT, output_filename)
    with open(output_path, "w", encoding="utf-8") as f:
        f.write(vtt_content)
    
    return output_path

def format_timestamp(seconds):
    minutes = int(seconds // 60)
    secs = int(seconds % 60)
    millis = int((seconds - int(seconds)) * 1000)
    return f"{minutes:02}:{secs:02}.{millis:03}"

def render_video(audio_path, avatar_image_path, output_filename, ai_text=None, lang='en'):
    """Generate video - uses static fallback since D-ID requires public URLs"""
    return render_static_video(audio_path, avatar_image_path, output_filename)

def render_static_video(audio_path, avatar_image_path, output_filename):
    """Fast static video generation"""
    audio_clip = AudioFileClip(audio_path)
    image_clip = ImageClip(avatar_image_path).set_duration(audio_clip.duration)
    image_clip = image_clip.set_audio(audio_clip)
    
    output_path = os.path.join(MEDIA_ROOT, output_filename)
    image_clip.write_videofile(
        output_path, 
        fps=15,
        codec='libx264', 
        audio_codec='aac', 
        preset='ultrafast',
        threads=4,
        logger=None,
        verbose=False
    )
    
    audio_clip.close()
    image_clip.close()
    
    return output_path

