"""
D-ID Clips API - Generates lip-synced videos without WebRTC
This works on any network and creates actual videos with mouth movement
"""
import requests
import os
import time
from django.conf import settings

def create_did_clip(text, output_path="static/videos/"):
    """
    Create a lip-synced video using D-ID Clips API
    Returns the video URL when ready
    """
    api_key = os.getenv('DID_API_KEY')
    
    # Create output directory if it doesn't exist
    os.makedirs(output_path, exist_ok=True)
    
    # D-ID Clips API endpoint
    url = "https://api.d-id.com/clips"
    
    headers = {
        "Authorization": f"Basic {api_key}",
        "Content-Type": "application/json"
    }
    
    # Use male avatar and voice
    payload = {
        "script": {
            "type": "text",
            "input": text,
            "provider": {
                "type": "microsoft",
                "voice_id": "en-US-GuyNeural"  # Male voice
            }
        },
        "config": {
            "result_format": "mp4"
        },
        "presenter_config": {
            "type": "clip",
            "clip_id": "jack-jcwCkr1grs"  # Male presenter
        },
        "driver_id": "uM00QMwJ9x"  # D-ID's default driver for natural movement
    }
    
    print(f"Creating D-ID clip for text: {text[:50]}...")
    
    # Create the clip
    response = requests.post(url, json=payload, headers=headers)
    
    if response.status_code != 201:
        print(f"❌ D-ID Clip creation failed: {response.status_code} - {response.text}")
        return None
    
    clip_data = response.json()
    clip_id = clip_data.get('id')
    
    print(f"✓ D-ID Clip created! ID: {clip_id}")
    print(f"Waiting for video generation...")
    
    # Poll for completion (D-ID generates video in background)
    max_attempts = 60  # 60 seconds max wait
    attempt = 0
    
    while attempt < max_attempts:
        time.sleep(1)
        attempt += 1
        
        # Check clip status
        status_url = f"https://api.d-id.com/clips/{clip_id}"
        status_response = requests.get(status_url, headers=headers)
        
        if status_response.status_code != 200:
            print(f"❌ Failed to check status: {status_response.text}")
            return None
        
        status_data = status_response.json()
        status = status_data.get('status')
        
        print(f"Status: {status} ({attempt}s)")
        
        if status == 'done':
            video_url = status_data.get('result_url')
            print(f"✓ Video ready! URL: {video_url}")
            
            # Return the D-ID URL directly for immediate playback
            # Don't download - let browser stream it directly
            return video_url
        
        elif status == 'error':
            error = status_data.get('error', {})
            print(f"❌ Video generation failed: {error}")
            return None
    
    print(f"❌ Timeout waiting for video generation")
    return None
