from django.shortcuts import render
from django.http import JsonResponse
from django.views.decorators.csrf import csrf_exempt
import json
import os
import time
import asyncio
from asgiref.sync import sync_to_async, async_to_sync
from .video_engine import (
    generate_audio, generate_subtitles, render_video,
    create_did_stream, start_did_stream, submit_did_ice, send_did_talk,
    generate_elevenlabs_audio, upload_to_fileio
)
from .rag import query_knowledge_base
from openai import OpenAI
from django.conf import settings

# Initialize OpenAI Client with client's API key
openai_api_key = os.environ.get("OPENAI_API_KEY", "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")
if openai_api_key:
    client = OpenAI(api_key=openai_api_key)
    print(f"✓ OpenAI client initialized successfully")
else:
    print("WARNING: OPENAI_API_KEY not set")
    client = None

def index(request):
    return render(request, 'index.html')

@csrf_exempt
def create_stream_view(request):
    if request.method == 'POST':
        try:
            # Try multiple reliable public avatar URLs
            # D-ID needs to be able to download these images
            
            avatar_urls = [
                # Option 1: Unsplash (very reliable CDN)
                "https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=400",
                
                # Option 2: Pexels (free stock photos)
                "https://images.pexels.com/photos/774909/pexels-photo-774909.jpeg?auto=compress&cs=tinysrgb&w=400",
                
                # Option 3: Cloudinary demo
                "https://res.cloudinary.com/demo/image/upload/w_400/sample.jpg",
                
                # Option 4: GitHub raw content
                "https://raw.githubusercontent.com/d-id/stable-diffusion-webui-d-id/main/example.png",
            ]
            
            # Use the first URL (Unsplash - most reliable)
            source_url = avatar_urls[0]
            
            print(f"Using source URL: {source_url}")
            
            data = create_did_stream(source_url)
            return JsonResponse(data)
        except Exception as e:
            print(f"Error in create_stream_view: {e}")
            return JsonResponse({'error': str(e)}, status=500)
    return JsonResponse({'error': 'Invalid method'}, status=405)

@csrf_exempt
def start_stream_view(request):
    if request.method == 'POST':
        try:
            data = json.loads(request.body)
            stream_id = data.get('stream_id')
            session_id = data.get('session_id')
            answer = data.get('answer')
            
            print(f"Starting stream: stream_id={stream_id}, session_id={session_id}")
            
            result = start_did_stream(stream_id, session_id, answer)
            return JsonResponse(result)
        except Exception as e:
            print(f"Error in start_stream_view: {e}")
            import traceback
            traceback.print_exc()
            return JsonResponse({'error': str(e)}, status=500)
    return JsonResponse({'error': 'Invalid method'}, status=405)

@csrf_exempt
def submit_ice_view(request):
    if request.method == 'POST':
        try:
            data = json.loads(request.body)
            stream_id = data.get('stream_id')
            session_id = data.get('session_id')
            candidate = data.get('candidate')
            sdp_mid = data.get('sdpMid')
            sdp_m_line_index = data.get('sdpMLineIndex')
            
            submit_did_ice(stream_id, session_id, candidate, sdp_mid, sdp_m_line_index)
            return JsonResponse({'status': 'ok'})
        except Exception as e:
            return JsonResponse({'error': str(e)}, status=500)
    return JsonResponse({'error': 'Invalid method'}, status=405)

@csrf_exempt
def chat_view(request):
    if request.method == 'POST':
        try:
            data = json.loads(request.body)
            user_text = data.get('text')
            stream_id = data.get('stream_id')
            session_id = data.get('session_id')
            
            print(f"Chat request: text='{user_text}', stream_id={stream_id}")
            
            # Check if OpenAI client is available
            if not client:
                return JsonResponse({'error': 'OpenAI API key not configured'}, status=500)
            
            # 1. Get AI Response using OpenAI
            try:
                rag_context = query_knowledge_base(user_text)
            except Exception as e:
                print(f"RAG query failed: {e}. Continuing without context.")
                rag_context = None
                
            system_prompt = "You are a helpful AI assistant. Provide clear, informative answers in 2-3 sentences. Be concise but comprehensive."
            if rag_context:
                system_prompt += f"\nContext: {rag_context[:200]}"

            # Call OpenAI API
            completion = client.chat.completions.create(
                model="gpt-4o-mini",
                messages=[
                    {"role": "system", "content": system_prompt},
                    {"role": "user", "content": user_text}
                ],
                max_tokens=150,  # Increased for longer answers
                temperature=0.7  # Slightly more creative
            )
            ai_answer = completion.choices[0].message.content
            
            print(f"AI Answer: {ai_answer}")
            
            # Generate lip-synced video using D-ID Clips API (works without WebRTC!)
            from agent.video_engine_clips import create_did_clip
            
            video_url = create_did_clip(ai_answer)
            
            if video_url:
                return JsonResponse({
                    'answer': ai_answer,
                    'video_url': video_url,
                    'status': 'success'
                })
            else:
                # Fallback if video generation fails
                return JsonResponse({
                    'answer': ai_answer,
                    'video_url': None,
                    'status': 'audio_only'
                })
            
        except Exception as e:
            print(f"Error in chat_view: {e}")
            import traceback
            traceback.print_exc()
            return JsonResponse({'error': str(e)}, status=500)
    return JsonResponse({'error': 'Invalid method'}, status=405)


@csrf_exempt
def generate_video_view(request):
    if request.method == 'POST':
        try:
            print("Received generate_video request (sync)")
            data = json.loads(request.body)
            text = data.get('text', '')
            lang = data.get('lang', 'en')
            
            if not text:
                return JsonResponse({'error': 'Text is required'}, status=400)

            timestamp = int(time.time())
            audio_filename = f"audio_{timestamp}.mp3"
            video_filename = f"video_{timestamp}.mp4"
            
            print(f"Generating audio for text: {text[:50]}...")
            # Run async generate_audio in sync view
            audio_path = async_to_sync(generate_audio)(text, lang, audio_filename)
            print(f"Audio generated at {audio_path}")
            
            avatar_path = os.path.join(settings.BASE_DIR, 'agent', 'static', 'avatar.png')
            
            # render_video is synchronous, so we can call it directly.
            # Since this view is sync, Django runs it in a thread, so it won't block the loop.
            print("Starting video rendering...")
            video_path = render_video(audio_path, avatar_path, video_filename)
            print(f"Video generated at {video_path}")
            
            video_url = f"/media/{video_filename}"
            
            return JsonResponse({'video_url': video_url})
        except Exception as e:
            print(f"Error in generate_video_view: {e}")
            return JsonResponse({'error': str(e)}, status=500)
    return JsonResponse({'error': 'Invalid method'}, status=405)
